const db = require("../config/db")

class TestCase {
  static async getAllTestCases() {
    try {
      const [rows] = await db.query(`
        SELECT 
          tc.*, 
          u.username as created_by,
          t.test_name,
          t.description as test_description
        FROM test_cases tc
        LEFT JOIN users u ON tc.user_id = u.id
        LEFT JOIN tests t ON tc.test_id = t.id
        ORDER BY tc.test_case_id
      `)
      return rows
    } catch (error) {
      console.error("Error in getAllTestCases:", error)
      throw error
    }
  }

  static async getTestCaseById(id) {
    try {
      const [rows] = await db.query(
        `
        SELECT 
          tc.*, 
          u.username as created_by,
          t.test_name,
          t.description as test_description
        FROM test_cases tc
        LEFT JOIN users u ON tc.user_id = u.id
        LEFT JOIN tests t ON tc.test_id = t.id
        WHERE tc.id = ?
      `,
        [id],
      )
      return rows[0] || null
    } catch (error) {
      console.error("Error in getTestCaseById:", error)
      throw error
    }
  }

  static async createTestCase(testCaseData) {
    try {
      const { 
        test_case_id, 
        description, 
        test_step, 
        test_data, 
        expected_result, 
        status, 
        framework, 
        user_id,
        test_id 
      } = testCaseData

      const [result] = await db.query(
        "INSERT INTO test_cases (test_case_id, description, test_step, test_data, expected_result, status, framework, user_id, test_id) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)",
        [test_case_id, description, test_step, test_data, expected_result, status, framework, user_id, test_id],
      )
      return result
    } catch (error) {
      console.error("Error in createTestCase:", error)
      throw error
    }
  }

  static async updateTestCase(id, testCaseData) {
    try {
      const { 
        test_case_id, 
        description, 
        test_step, 
        test_data, 
        expected_result, 
        status, 
        framework,
        test_id 
      } = testCaseData

      const [result] = await db.query(
        "UPDATE test_cases SET test_case_id = ?, description = ?, test_step = ?, test_data = ?, expected_result = ?, status = ?, framework = ?, test_id = ? WHERE id = ?",
        [test_case_id, description, test_step, test_data, expected_result, status, framework, test_id, id],
      )
      return result
    } catch (error) {
      console.error("Error in updateTestCase:", error)
      throw error
    }
  }

  static async deleteTestCase(id) {
    try {
      const [result] = await db.query("DELETE FROM test_cases WHERE id = ?", [id])
      return result
    } catch (error) {
      console.error("Error in deleteTestCase:", error)
      throw error
    }
  }

  static async searchTestCases(searchTerm) {
    try {
      const [rows] = await db.query(
        `SELECT 
          tc.*, 
          u.username as created_by,
          t.test_name,
          t.description as test_description
         FROM test_cases tc
         LEFT JOIN users u ON tc.user_id = u.id
         LEFT JOIN tests t ON tc.test_id = t.id
         WHERE tc.test_case_id LIKE ? 
         OR tc.description LIKE ? 
         OR tc.test_step LIKE ?
         OR t.test_name LIKE ?`,
        [`%${searchTerm}%`, `%${searchTerm}%`, `%${searchTerm}%`, `%${searchTerm}%`],
      )
      return rows
    } catch (error) {
      console.error("Error in searchTestCases:", error)
      throw error
    }
  }

  static async getTestCasesByFramework(framework) {
    try {
      const [rows] = await db.query(
        `SELECT 
          tc.*, 
          u.username as created_by,
          t.test_name,
          t.description as test_description
         FROM test_cases tc
         LEFT JOIN users u ON tc.user_id = u.id
         LEFT JOIN tests t ON tc.test_id = t.id
         WHERE tc.framework = ? 
         ORDER BY tc.test_case_id`,
        [framework],
      )
      return rows
    } catch (error) {
      console.error("Error in getTestCasesByFramework:", error)
      throw error
    }
  }

  static async getTestCasesByStatus(status) {
    try {
      const [rows] = await db.query(
        `SELECT 
          tc.*, 
          u.username as created_by,
          t.test_name,
          t.description as test_description
         FROM test_cases tc
         LEFT JOIN users u ON tc.user_id = u.id
         LEFT JOIN tests t ON tc.test_id = t.id
         WHERE tc.status = ? 
         ORDER BY tc.test_case_id`,
        [status],
      )
      return rows
    } catch (error) {
      console.error("Error in getTestCasesByStatus:", error)
      throw error
    }
  }

  static async getTestCasesByTestId(testId) {
    try {
      const [rows] = await db.query(
        `SELECT 
          tc.*, 
          u.username as created_by,
          t.test_name,
          t.description as test_description
         FROM test_cases tc
         LEFT JOIN users u ON tc.user_id = u.id
         LEFT JOIN tests t ON tc.test_id = t.id
         WHERE tc.test_id = ? 
         ORDER BY tc.test_case_id`,
        [testId],
      )
      return rows
    } catch (error) {
      console.error("Error in getTestCasesByTestId:", error)
      throw error
    }
  }
}

module.exports = TestCase

