"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __param = (this && this.__param) || function (paramIndex, decorator) {
    return function (target, key) { decorator(target, key, paramIndex); }
};
var ProductionTargetsService_1;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProductionTargetsService = void 0;
const common_1 = require("@nestjs/common");
const typeorm_1 = require("@nestjs/typeorm");
const typeorm_2 = require("typeorm");
const production_target_entity_1 = require("./entities/production_target.entity");
const order_entity_1 = require("../orders/entities/order.entity");
const product_entity_1 = require("../products/entities/product.entity");
const material_entity_1 = require("../materials/entities/material.entity");
const product_stock_entity_1 = require("../product_stocks/entities/product_stock.entity");
const material_stock_entity_1 = require("../material_stocks/entities/material_stock.entity");
let ProductionTargetsService = ProductionTargetsService_1 = class ProductionTargetsService {
    constructor(productionTargetRepository, orderRepository, productRepository, materialRepository, productStockRepository, materialStockRepository) {
        this.productionTargetRepository = productionTargetRepository;
        this.orderRepository = orderRepository;
        this.productRepository = productRepository;
        this.materialRepository = materialRepository;
        this.productStockRepository = productStockRepository;
        this.materialStockRepository = materialStockRepository;
        this.logger = new common_1.Logger(ProductionTargetsService_1.name);
    }
    async create(createProductionTargetDto) {
        this.logger.log(`🔄 [START] Creating/Updating production target for itemID: ${createProductionTargetDto.itemID}`);
        const { ProductionTargetID, OrderID, itemID, itemType, ActualProduced, Status, startTime, endTime, ...targetData } = createProductionTargetDto;
        if (!Number.isInteger(itemID) || itemID <= 0) {
            this.logger.warn(`⚠️ Invalid itemID received: ${itemID}`);
            throw new common_1.BadRequestException(`Invalid itemID: ${itemID}`);
        }
        let order = null;
        if (OrderID !== undefined && OrderID !== null && OrderID !== -1) {
            const foundOrder = await this.orderRepository.findOne({
                where: { OrderID },
            });
            if (!foundOrder) {
                throw new common_1.NotFoundException(`Order with ID ${OrderID} not found`);
            }
            order = foundOrder;
        }
        if (itemType === 'PRODUCT') {
            const productExists = await this.productRepository.exist({
                where: { ProductID: itemID },
            });
            if (!productExists) {
                throw new common_1.NotFoundException(`Product with ID ${itemID} not found`);
            }
        }
        else if (itemType === 'MATERIAL') {
            const materialExists = await this.materialRepository.exist({
                where: { MaterialID: itemID },
            });
            if (!materialExists) {
                throw new common_1.NotFoundException(`Material with ID ${itemID} not found`);
            }
        }
        else {
            throw new common_1.BadRequestException(`Invalid itemType: ${itemType}`);
        }
        if (ProductionTargetID) {
            const existingTarget = await this.productionTargetRepository.findOne({
                where: { ProductionTargetID },
                relations: ['order'],
            });
            if (existingTarget) {
                if (targetData.TargetProduced !== undefined) {
                    existingTarget.TargetProduced = targetData.TargetProduced;
                }
                if (ActualProduced !== undefined) {
                    existingTarget.ActualProduced = ActualProduced;
                }
                if (Status !== undefined) {
                    existingTarget.Status = Status;
                }
                if (startTime !== undefined) {
                    existingTarget.startTime = startTime;
                }
                if (endTime !== undefined) {
                    existingTarget.endTime = endTime;
                }
                if (order) {
                    existingTarget.order = order;
                }
                existingTarget.itemID = itemID;
                existingTarget.itemType = itemType;
                await this.productionTargetRepository.save(existingTarget);
                await this.updateStatusAndStock(existingTarget, null);
                await this.calculateProductionHours(existingTarget.ProductionTargetID);
                return await this.productionTargetRepository.findOne({
                    where: { ProductionTargetID },
                    relations: ['order'],
                });
            }
        }
        const finalStatus = Status ?? 'กำลังรอ';
        const newTarget = this.productionTargetRepository.create({
            ...targetData,
            itemID,
            itemType,
            ActualProduced: ActualProduced ?? 0,
            Status: finalStatus,
            order,
            startTime: startTime ?? null,
            endTime: endTime ?? null,
        });
        const savedTarget = await this.productionTargetRepository.save(newTarget);
        await this.updateStatusAndStock(savedTarget, null);
        await this.calculateProductionHours(savedTarget.ProductionTargetID);
        return await this.productionTargetRepository.findOne({
            where: { ProductionTargetID: savedTarget.ProductionTargetID },
            relations: ['order'],
        });
    }
    async createMany(createManyDto) {
        this.logger.log(`📦 [START] Creating or Updating ${createManyDto.productionTargets.length} production targets`);
        const validTargets = [];
        for (const [index, targetDto,] of createManyDto.productionTargets.entries()) {
            try {
                this.logger.log(`➡️ [${index + 1}] Processing target with itemID: ${targetDto.itemID}`);
                const newOrUpdatedTarget = await this.create(targetDto);
                validTargets.push(newOrUpdatedTarget);
            }
            catch (error) {
                this.logger.error(`❌ [${index + 1}] Failed to process target: ${error.message}`, error.stack);
            }
        }
        this.logger.log(`✅ [DONE] Successfully processed ${validTargets.length} production targets`);
        return validTargets;
    }
    async findAll() {
        this.logger.log(`Retrieving all production targets`);
        const targets = await this.productionTargetRepository.find({
            relations: ['order'],
        });
        const [products, materials] = await Promise.all([
            this.productRepository.find(),
            this.materialRepository.find(),
        ]);
        return targets.map((target) => {
            if (target.itemType === 'PRODUCT') {
                const product = products.find((p) => p.ProductID === target.itemID);
                return {
                    ...target,
                    item: product
                        ? {
                            id: product.ProductID,
                            type: 'PRODUCT',
                            brand: product.brand,
                            size: product.size,
                            unit: product.unit,
                            quantityInStock: product.quantityInStock,
                            lowStockLevel: product.lowStockLevel,
                            pricePerUnit: product.pricePerUnit,
                            status: product.status,
                        }
                        : null,
                };
            }
            else if (target.itemType === 'MATERIAL') {
                const material = materials.find((m) => m.MaterialID === target.itemID);
                return {
                    ...target,
                    item: material
                        ? {
                            id: material.MaterialID,
                            type: 'MATERIAL',
                            name: material.name,
                            brand: material.brand,
                            size: material.size,
                            unit: material.unit,
                            quantityInStock: material.quantityInStock,
                            lowStockLevel: material.lowStockLevel,
                            pricePerUnit: material.pricePerUnit,
                            status: material.status,
                            lastUpdate: material.LastUpdate,
                        }
                        : null,
                };
            }
            return {
                ...target,
                item: null,
            };
        });
    }
    async findOne(id) {
        this.logger.log(`Retrieving production target with ID: ${id}`);
        const target = await this.productionTargetRepository.findOne({
            where: { ProductionTargetID: id },
            relations: ['order'],
        });
        if (!target) {
            throw new common_1.NotFoundException(`ProductionTarget with ID ${id} not found`);
        }
        return target;
    }
    async findByDate(dateStr) {
        this.logger.log(`Retrieving production targets for date: ${dateStr}`);
        if (!dateStr)
            throw new common_1.BadRequestException('Date is required');
        const dateParts = dateStr.trim().split('/');
        if (dateParts.length !== 3 ||
            dateParts.some((part) => isNaN(Number(part)))) {
            throw new common_1.BadRequestException('Invalid date format. Expected "dd/mm/yyyy"');
        }
        const [day, month, year] = dateParts.map(Number);
        const startOfDay = new Date(year, month - 1, day, 0, 0, 0);
        const endOfDay = new Date(year, month - 1, day, 23, 59, 59);
        return this.productionTargetRepository.find({
            where: { Date: (0, typeorm_2.Between)(startOfDay, endOfDay) },
        });
    }
    async update(id, updateProductionTargetDto) {
        this.logger.log(`Updating production target with ID: ${id}`);
        const { OrderID, itemID, itemType, startTime, endTime, ...rest } = updateProductionTargetDto;
        const target = await this.productionTargetRepository.findOne({
            where: { ProductionTargetID: id },
            relations: ['order'],
        });
        if (!target) {
            throw new common_1.NotFoundException(`ProductionTarget with ID ${id} not found`);
        }
        if (OrderID !== undefined && OrderID !== null) {
            const order = await this.orderRepository.findOne({
                where: { OrderID },
            });
            if (!order) {
                throw new common_1.NotFoundException(`Order with ID ${OrderID} not found`);
            }
            target.order = order;
        }
        if (itemID && itemType) {
            if (!Number.isInteger(itemID) || itemID <= 0) {
                throw new common_1.BadRequestException(`Invalid itemID: ${itemID}`);
            }
            let itemExists = false;
            if (itemType === 'PRODUCT') {
                itemExists = await this.productRepository.exist({
                    where: { ProductID: itemID },
                });
            }
            else if (itemType === 'MATERIAL') {
                itemExists = await this.materialRepository.exist({
                    where: { MaterialID: itemID },
                });
            }
            else {
                throw new common_1.BadRequestException(`Invalid itemType: ${itemType}`);
            }
            if (!itemExists) {
                throw new common_1.NotFoundException(`Item with ID ${itemID} and Type ${itemType} not found`);
            }
            target.itemID = itemID;
            target.itemType = itemType;
        }
        if (startTime !== undefined) {
            target.startTime = startTime;
        }
        if (endTime !== undefined) {
            target.endTime = endTime;
        }
        Object.assign(target, rest);
        const updatedTarget = await this.productionTargetRepository.save(target);
        await this.updateStatusAndStock(updatedTarget, target.ActualProduced);
        await this.calculateProductionHours(updatedTarget.ProductionTargetID);
        return updatedTarget;
    }
    async remove(id) {
        this.logger.log(`Removing production target with ID: ${id}`);
        const target = await this.findOne(id);
        if (target.ActualProduced && target.ActualProduced > 0) {
            await this.rollbackStock(target);
        }
        return await this.productionTargetRepository.remove(target);
    }
    async calculateProductionHours(productionTargetID) {
        const target = await this.productionTargetRepository.findOne({
            where: { ProductionTargetID: productionTargetID },
        });
        if (!target)
            return;
        if (target.startTime && target.endTime) {
            const hours = (target.endTime.getTime() - target.startTime.getTime()) /
                (1000 * 60 * 60);
            target.totalProductionHours = parseFloat(hours.toFixed(2));
        }
        else {
            target.totalProductionHours = null;
        }
        await this.productionTargetRepository.save(target);
    }
    async updateStatusAndStock(target, oldActualProduced) {
        const difference = target.TargetProduced - target.ActualProduced;
        if (difference === 0) {
            target.Status = 'พอดี';
        }
        else if (difference > 0) {
            target.Status = `ขาด ${difference}`;
        }
        else {
            target.Status = `เกิน ${Math.abs(difference)}`;
        }
        await this.productionTargetRepository.save(target);
        const oldVal = oldActualProduced ?? 0;
        const newVal = target.ActualProduced;
        const diffProduced = newVal - oldVal;
        if (diffProduced !== 0) {
            if (target.itemType === 'PRODUCT') {
                await this.updateProductStock(target.itemID, diffProduced);
            }
            else {
                await this.updateMaterialStock(target.itemID, diffProduced);
            }
        }
    }
    async updateProductStock(productID, diff) {
        const product = await this.productRepository.findOne({
            where: { ProductID: productID },
        });
        if (!product) {
            this.logger.error(`Product not found for ID: ${productID}`);
            return;
        }
        product.quantityInStock += diff;
        await this.productRepository.save(product);
        const stockRecord = new product_stock_entity_1.ProductStock();
        stockRecord.product = product;
        stockRecord.quantityReceived = diff;
        stockRecord.status = 'เติมStock';
        stockRecord.addDate = new Date();
        await this.productStockRepository.save(stockRecord);
        this.logger.log(`Updated product(ID: ${productID}) stock by ${diff} → ${product.quantityInStock}`);
    }
    async updateMaterialStock(materialID, diff) {
        const material = await this.materialRepository.findOne({
            where: { MaterialID: materialID },
        });
        if (!material) {
            this.logger.error(`Material not found for ID: ${materialID}`);
            return;
        }
        material.quantityInStock += diff;
        await this.materialRepository.save(material);
        const materialStock = new material_stock_entity_1.MaterialStock();
        materialStock.material = material;
        materialStock.QuantityOrdered = 0;
        materialStock.QuantityReceived = diff;
        materialStock.Status = 'เติมStock';
        await this.materialStockRepository.save(materialStock);
        this.logger.log(`Updated material(ID: ${materialID}) stock by ${diff} → ${material.quantityInStock}`);
    }
    async rollbackStock(target) {
        this.logger.log(`Start to rollback for ProductionTargetID: ${target.ProductionTargetID}`);
        const rollbackAmount = target.ActualProduced;
        if (!rollbackAmount || rollbackAmount === 0) {
            this.logger.log(`ℹ️ No stock to rollback`);
            return;
        }
        if (target.itemType === 'PRODUCT') {
            await this.updateProductStock(target.itemID, -rollbackAmount);
        }
        else {
            await this.updateMaterialStock(target.itemID, -rollbackAmount);
        }
        this.logger.log(`✅ Rollback stock completed for ProductionTargetID: ${target.ProductionTargetID}`);
    }
    async deleteProductionTargetsFromDate(dateStr) {
        this.logger.log(`🗑️ Deleting production targets from date: ${dateStr}`);
        if (!dateStr)
            throw new common_1.BadRequestException('dateStr is required');
        let startDate;
        const parts = dateStr.includes('/')
            ? dateStr.trim().split('/')
            : dateStr.trim().split('-');
        if (parts.length !== 3) {
            throw new common_1.BadRequestException('Invalid date format. Use dd/MM/yyyy or yyyy-MM-dd');
        }
        if (dateStr.includes('/')) {
            const [day, month, year] = parts.map(Number);
            startDate = new Date(year, month - 1, day);
        }
        else {
            const [year, month, day] = parts.map(Number);
            startDate = new Date(year, month - 1, day);
        }
        startDate.setHours(0, 0, 0, 0);
        const targetsToDelete = await this.productionTargetRepository.find({
            where: { Date: (0, typeorm_2.MoreThanOrEqual)(startDate) },
        });
        if (targetsToDelete.length === 0) {
            this.logger.warn(`⚠️ No production targets found from ${dateStr} onwards`);
            return { deletedCount: 0 };
        }
        this.logger.log(`📌 Production Targets to be deleted (${targetsToDelete.length} items):`);
        targetsToDelete.forEach((target) => {
            this.logger.log(`🗑️ TargetID: ${target.ProductionTargetID}, Date: ${target.Date}`);
        });
        await this.productionTargetRepository.remove(targetsToDelete);
        this.logger.log(`✅ Successfully deleted ${targetsToDelete.length} production targets.`);
        return { deletedCount: targetsToDelete.length };
    }
};
exports.ProductionTargetsService = ProductionTargetsService;
exports.ProductionTargetsService = ProductionTargetsService = ProductionTargetsService_1 = __decorate([
    (0, common_1.Injectable)(),
    __param(0, (0, typeorm_1.InjectRepository)(production_target_entity_1.ProductionTarget)),
    __param(1, (0, typeorm_1.InjectRepository)(order_entity_1.Order)),
    __param(2, (0, typeorm_1.InjectRepository)(product_entity_1.Product)),
    __param(3, (0, typeorm_1.InjectRepository)(material_entity_1.Material)),
    __param(4, (0, typeorm_1.InjectRepository)(product_stock_entity_1.ProductStock)),
    __param(5, (0, typeorm_1.InjectRepository)(material_stock_entity_1.MaterialStock)),
    __metadata("design:paramtypes", [typeorm_2.Repository,
        typeorm_2.Repository,
        typeorm_2.Repository,
        typeorm_2.Repository,
        typeorm_2.Repository,
        typeorm_2.Repository])
], ProductionTargetsService);
//# sourceMappingURL=production_targets.service.js.map