"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __param = (this && this.__param) || function (paramIndex, decorator) {
    return function (target, key) { decorator(target, key, paramIndex); }
};
var QueuesService_1;
Object.defineProperty(exports, "__esModule", { value: true });
exports.QueuesService = void 0;
const common_1 = require("@nestjs/common");
const typeorm_1 = require("@nestjs/typeorm");
const typeorm_2 = require("typeorm");
const queue_entity_1 = require("./entities/queue.entity");
const machine_entity_1 = require("../machines/entities/machine.entity");
const page_entity_1 = require("../pages/entities/page.entity");
const order_entity_1 = require("../orders/entities/order.entity");
const employee_entity_1 = require("../employees/entities/employee.entity");
let QueuesService = QueuesService_1 = class QueuesService {
    deleteByDate(dateStr) {
        throw new Error('Method not implemented.');
    }
    constructor(queueRepository, machineRepository, pageRepository, orderRepository, employeeRepository) {
        this.queueRepository = queueRepository;
        this.machineRepository = machineRepository;
        this.pageRepository = pageRepository;
        this.orderRepository = orderRepository;
        this.employeeRepository = employeeRepository;
        this.logger = new common_1.Logger(QueuesService_1.name);
    }
    async getLatestQueuesFromDate(dateStr) {
        this.logger.log(`📡 Fetching queues from ${dateStr || 'the most recent date'}`);
        let startDate;
        if (dateStr) {
            const parts = dateStr.includes('/')
                ? dateStr.trim().split('/')
                : dateStr.trim().split('-');
            if (parts.length !== 3) {
                throw new common_1.BadRequestException('Invalid date format. Use dd/MM/yyyy or yyyy-MM-dd');
            }
            if (dateStr.includes('/')) {
                const [day, month, year] = parts.map(Number);
                startDate = new Date(year, month - 1, day);
            }
            else {
                const [year, month, day] = parts.map(Number);
                startDate = new Date(year, month - 1, day);
            }
        }
        else {
            const latestExistingQueue = await this.queueRepository.find({
                order: { createdAt: 'DESC' },
                take: 1,
            });
            if (!latestExistingQueue.length) {
                this.logger.warn(`⚠️ No queues found in the system`);
                return [];
            }
            startDate = new Date(latestExistingQueue[0].createdAt);
        }
        startDate.setHours(0, 0, 0, 0);
        this.logger.log(`🔍 Fetching all queues from ${startDate.toISOString()} onwards`);
        return await this.queueRepository.find({
            where: { createdAt: (0, typeorm_2.MoreThanOrEqual)(startDate) },
            order: { createdAt: 'ASC' },
        });
    }
    async create(createQueueDto) {
        const { MachineID, PageID, OrderID, EmployeeIds, ...queueData } = createQueueDto;
        const machine = await this.machineRepository.findOne({
            where: { MachineID },
        });
        if (!machine)
            throw new common_1.NotFoundException(`Machine with ID ${MachineID} not found`);
        const page = await this.pageRepository.findOne({ where: { PageID } });
        if (!page)
            throw new common_1.NotFoundException(`Page with ID ${PageID} not found`);
        const order = await this.orderRepository.findOne({ where: { OrderID } });
        if (!order)
            throw new common_1.NotFoundException(`Order with ID ${OrderID} not found`);
        const employees = await this.employeeRepository.findByIds(EmployeeIds);
        if (employees.length !== EmployeeIds.length) {
            throw new common_1.NotFoundException(`One or more employees not found`);
        }
        const newQueue = this.queueRepository.create({
            ...queueData,
            machine,
            page,
            order,
            employees,
        });
        const savedQueue = await this.queueRepository.save(newQueue);
        return {
            message: 'Queue created successfully',
            queue: savedQueue,
        };
    }
    async createMultiple(createQueueDtos) {
        const newQueues = [];
        if (!Array.isArray(createQueueDtos)) {
            throw new common_1.BadRequestException('Request body must be an array of queues');
        }
        for (const createQueueDto of createQueueDtos) {
            const { MachineID, PageID, OrderID, EmployeeIds, ...queueData } = createQueueDto;
            if (!Array.isArray(EmployeeIds) || EmployeeIds.length === 0) {
                throw new common_1.BadRequestException(`EmployeeIds must be a non-empty array`);
            }
            const validEmployeeIds = EmployeeIds.map((id) => Number(id)).filter((id) => !isNaN(id) && id > 0);
            if (validEmployeeIds.length !== EmployeeIds.length) {
                throw new common_1.BadRequestException(`Invalid EmployeeIds provided: ${EmployeeIds}`);
            }
            const machineId = Number(MachineID);
            if (isNaN(machineId))
                throw new common_1.BadRequestException(`Invalid MachineID: ${MachineID}`);
            const orderId = Number(OrderID);
            if (isNaN(orderId))
                throw new common_1.BadRequestException(`Invalid OrderID: ${OrderID}`);
            let pageId = null;
            if (PageID !== null && PageID !== undefined) {
                pageId = Number(PageID);
                if (isNaN(pageId))
                    throw new common_1.BadRequestException(`Invalid PageID: ${PageID}`);
            }
            const machine = await this.machineRepository.findOne({
                where: { MachineID: machineId },
            });
            if (!machine)
                throw new common_1.NotFoundException(`Machine with ID ${machineId} not found`);
            const page = pageId !== null
                ? await this.pageRepository.findOne({ where: { PageID: pageId } })
                : null;
            if (pageId !== null && !page)
                throw new common_1.NotFoundException(`Page with ID ${pageId} not found`);
            const order = await this.orderRepository.findOne({
                where: { OrderID: orderId },
            });
            if (!order)
                throw new common_1.NotFoundException(`Order with ID ${orderId} not found`);
            const employees = await this.employeeRepository.findByIds(validEmployeeIds);
            const foundIds = employees.map((e) => e.EmployeeID);
            const missingIds = validEmployeeIds.filter((id) => !foundIds.includes(id));
            if (missingIds.length > 0) {
                console.error(`❌ Missing EmployeeIDs: ${missingIds.join(', ')}`);
                throw new common_1.NotFoundException(`Employees not found: ${missingIds.join(', ')}`);
            }
            const newQueue = this.queueRepository.create({
                ...queueData,
                machine,
                page,
                order,
                employees,
            });
            newQueues.push(newQueue);
        }
        const savedQueues = await this.queueRepository.save(newQueues);
        return {
            message: `${savedQueues.length} queues created successfully`,
            queues: savedQueues,
        };
    }
    async findAll() {
        return await this.queueRepository.find({
            relations: ['machine', 'page', 'order', 'employees'],
        });
    }
    async findOne(id) {
        const queue = await this.queueRepository.findOne({
            where: { QueueID: id },
            relations: ['machine', 'page', 'order', 'employees'],
        });
        if (!queue) {
            throw new common_1.NotFoundException(`Queue with ID ${id} not found`);
        }
        return queue;
    }
    async update(id, updateQueueDto) {
        try {
            console.log('🔄 Updating Queue ID:', id, updateQueueDto);
            const { MachineID, PageID, OrderID, EmployeeIds, startTime, finishTime, ...queueData } = updateQueueDto;
            const existingQueue = await this.queueRepository.findOne({
                where: { QueueID: id },
                relations: ['machine', 'page', 'order', 'employees'],
            });
            if (!existingQueue) {
                throw new common_1.NotFoundException(`Queue with ID ${id} not found`);
            }
            const machine = MachineID
                ? await this.machineRepository.findOne({ where: { MachineID } })
                : existingQueue.machine;
            if (MachineID && !machine)
                throw new common_1.NotFoundException(`Machine with ID ${MachineID} not found`);
            const page = PageID
                ? await this.pageRepository.findOne({ where: { PageID } })
                : existingQueue.page;
            if (PageID && !page)
                throw new common_1.NotFoundException(`Page with ID ${PageID} not found`);
            const order = OrderID
                ? await this.orderRepository.findOne({ where: { OrderID } })
                : existingQueue.order;
            if (OrderID && !order)
                throw new common_1.NotFoundException(`Order with ID ${OrderID} not found`);
            let employees = existingQueue.employees;
            if (EmployeeIds) {
                const fetchedEmployees = await this.employeeRepository.findByIds(EmployeeIds);
                if (fetchedEmployees.length !== EmployeeIds.length) {
                    throw new common_1.NotFoundException(`One or more employees not found`);
                }
                employees = fetchedEmployees;
            }
            const updateData = {
                ...queueData,
                machine,
                page,
                order,
                employees,
                startTime: startTime ? new Date(startTime) : existingQueue.startTime,
                finishTime: finishTime
                    ? new Date(finishTime)
                    : existingQueue.finishTime,
            };
            const updatedQueue = await this.queueRepository.save({
                ...existingQueue,
                ...updateData,
            });
            console.log('✅ Queue updated successfully:', updatedQueue);
            return updatedQueue;
        }
        catch (error) {
            console.error('❌ Error in updateQueue:', error);
            throw new Error('Database update failed');
        }
    }
    async remove(id) {
        const queue = await this.queueRepository.findOne({
            where: { QueueID: id },
            relations: ['machine', 'page', 'order', 'employees'],
        });
        if (!queue) {
            throw new common_1.NotFoundException(`Queue with ID ${id} not found`);
        }
        await this.queueRepository.delete(id);
        return { message: 'Queue deleted successfully', deletedQueue: queue };
    }
    async deleteNewlyCreatedQueuesFromDate(dateStr) {
        this.logger.log(`🔄 Deleting newly created queues from date: ${dateStr}`);
        if (!dateStr)
            throw new common_1.BadRequestException('dateStr is required');
        let date;
        const parts = dateStr.includes('/')
            ? dateStr.trim().split('/')
            : dateStr.trim().split('-');
        if (parts.length !== 3) {
            throw new common_1.BadRequestException('Invalid date format. Use dd/MM/yyyy or yyyy-MM-dd');
        }
        if (dateStr.includes('/')) {
            const [day, month, year] = parts.map(Number);
            date = new Date(year, month - 1, day);
        }
        else {
            const [year, month, day] = parts.map(Number);
            date = new Date(year, month - 1, day);
        }
        const startOfDay = new Date(date.setHours(0, 0, 0, 0));
        const now = new Date();
        const queuesToDelete = await this.queueRepository.find({
            where: {
                startTime: (0, typeorm_2.MoreThanOrEqual)(startOfDay),
                createdAt: (0, typeorm_2.MoreThanOrEqual)(startOfDay),
            },
        });
        if (queuesToDelete.length === 0) {
            this.logger.warn(`⚠️ No queues matched deletion criteria for ${dateStr}`);
            return { deletedCount: 0 };
        }
        this.logger.log(`📌 Queues to be deleted (${queuesToDelete.length} items):`);
        queuesToDelete.forEach((queue) => {
            this.logger.log(`🗑️ QueueID: ${queue.QueueID}, startTime: ${queue.startTime}, createdAt: ${queue.createdAt}`);
        });
        await this.queueRepository.remove(queuesToDelete);
        this.logger.log(`✅ Successfully deleted ${queuesToDelete.length} queue(s).`);
        return { deletedCount: queuesToDelete.length };
    }
};
exports.QueuesService = QueuesService;
exports.QueuesService = QueuesService = QueuesService_1 = __decorate([
    (0, common_1.Injectable)(),
    __param(0, (0, typeorm_1.InjectRepository)(queue_entity_1.Queue)),
    __param(1, (0, typeorm_1.InjectRepository)(machine_entity_1.Machine)),
    __param(2, (0, typeorm_1.InjectRepository)(page_entity_1.Page)),
    __param(3, (0, typeorm_1.InjectRepository)(order_entity_1.Order)),
    __param(4, (0, typeorm_1.InjectRepository)(employee_entity_1.Employee)),
    __metadata("design:paramtypes", [typeorm_2.Repository,
        typeorm_2.Repository,
        typeorm_2.Repository,
        typeorm_2.Repository,
        typeorm_2.Repository])
], QueuesService);
//# sourceMappingURL=queues.service.js.map