import simpy
from datetime import datetime, timedelta
import math
from fastapi import FastAPI
from fastapi.middleware.cors import CORSMiddleware

# ==============================================
# 1) ข้อมูลเครื่องจักร (machines_info)
# ==============================================
machines_info = [
    {
        "machine_id": "MC001",
        "machine_name": "เครื่องเป่าขวด",
        "function": "เป่า",
        "materials": ["Preform"],  # ใช้วัตถุดิบ Preform
        "bottle_size": ["600 ml", "1500 ml"],
        "speed": [2200, 1100],  # 600 ml => 2200 ขวด/ชม, 1500 ml => 1100 ขวด/ชม
        "changeover_time": [3, 3],
        "status": "พร้อมใช้งาน"
    },
    {
        "machine_id": "MC002",
        "machine_name": "เครื่องเป่าขวด 350ml",
        "function": "เป่า",
        "materials": ["Preform"],
        "bottle_size": ["350 ml"],
        "speed": [2000],
        "changeover_time": [0],
        "status": "พร้อมใช้งาน"
    },
    {
        "machine_id": "MC003",
        "machine_name": "เครื่องสวมฉลาก",
        "function": "สวมฉลาก",
        "materials": ["Label", "EmptyBottle"],
        "bottle_size": ["350 ml", "600 ml", "1500 ml"],
        "speed": [4000, 4000, 2000],
        "changeover_time": [0, 0, 0],
        "status": "พร้อมใช้งาน"
    },
    {
        "machine_id": "MC004",
        "machine_name": "เครื่องบรรจุ + แพ็ค",
        "function": "บรรจุ, แพ็ค",
        "materials": ["EmptyBottleLabeled", "Cap", "PE_film"],
        "bottle_size": ["350 ml", "600 ml", "1500 ml"],
        "speed": [2400, 2400, 2400],
        "changeover_time": [0.5, 0.5, 0.5],
        "status": "พร้อมใช้งาน"
    }
]

# ==============================================
# 2) สต็อกวัตถุดิบ
# ==============================================
raw_material_stock = {
    "Preform": 20000,
    "Label_A": 20000,
    "Label_B": 20000,
    "Cap_A":   20000,
    "Cap_B":   20000,
    # 1 roll => 9600 ขวด
    "PE_film_roll": 20
}

# ==============================================
# 3) Orders ลูกค้า (ตัวอย่าง)
# ==============================================
customer_orders = [
    {
        "customer": "Wittaya",  
        "type": "water",
        "size": "600 ml",
        "brand": "A",
        "quantity_bottles": 8400,
        "due_date": "2025-01-08",
        "priority": 2
    },
    {
        "customer": "Aof",
        "type": "water",
        "size": "600 ml",
        "brand": "A",
        "quantity_bottles": 3600,
        "due_date": "2025-01-07",
        "priority": 1
    }
]

# ==============================================
# 4) ตาราง config Buffer Stock
# ==============================================
buffer_config = [
    {"type": "water",  "size": "350 ml",  "brand": "A", "want_to_store": 0,    "current_store": 0},
    {"type": "water",  "size": "600 ml",  "brand": "A", "want_to_store": 8000, "current_store": 0},
    {"type": "water",  "size": "1500 ml", "brand": "A", "want_to_store": 6000, "current_store": 0},
    {"type": "water",  "size": "350 ml",  "brand": "B", "want_to_store": 0,    "current_store": 0},
    {"type": "water",  "size": "600 ml",  "brand": "B", "want_to_store": 6000, "current_store": 0},
    {"type": "water",  "size": "1500 ml", "brand": "B", "want_to_store": 6000, "current_store": 0},
    {"type": "bottle", "size": "350 ml",  "brand": None,"want_to_store": 0,    "current_store": 0},
    {"type": "bottle", "size": "600 ml",  "brand": None,"want_to_store": 6000, "current_store": 0},
    {"type": "bottle", "size": "1500 ml", "brand": None,"want_to_store": 6000, "current_store": 0}
]

# ==============================================
# 5) ตาราง priority ของการผลิตเผื่อ
# ==============================================
buffer_priority = [
    {"priority": 1, "type": "water",  "size": "600 ml",  "brand": "A"},
    {"priority": 2, "type": "water",  "size": "1500 ml", "brand": "A"},
    {"priority": 3, "type": "water",  "size": "350 ml",  "brand": "A"},
    {"priority": 4, "type": "water",  "size": "600 ml",  "brand": "B"},
    {"priority": 5, "type": "water",  "size": "1500 ml", "brand": "B"},
    {"priority": 6, "type": "water",  "size": "350 ml",  "brand": "B"},
    {"priority": 7, "type": "bottle", "size": "600 ml",  "brand": None},
    {"priority": 8, "type": "bottle", "size": "1500 ml", "brand": None},
    {"priority": 9, "type": "bottle", "size": "350 ml",  "brand": None}
]

# ==============================================
# 6) เวลา และ Block การทำงาน
# ==============================================
WORK_START_1 = 8
WORK_END_1   = 12
WORK_START_2 = 13
WORK_END_2   = 17
MIN_BLOCK_LEFT = 0.5  # ถ้าบล็อกเหลือ < 0.5 ชม => ข้าม

def get_current_datetime(env):
    start_sim = datetime(2025, 1, 1, 8, 0, 0)
    return start_sim + timedelta(hours=env.now)

def skip_to_next_work_time(env):
    while True:
        now_dt = get_current_datetime(env)
        wd = now_dt.weekday()
        hr = now_dt.hour
        if wd >= 5:  # เสาร์/อาทิตย์
            days_to_monday = 7 - wd
            next_mon_8 = datetime(now_dt.year, now_dt.month, now_dt.day, 8) + timedelta(days=days_to_monday)
            yield env.timeout((next_mon_8 - now_dt).total_seconds()/3600)
            continue

        if hr < WORK_START_1:
            next_8 = datetime(now_dt.year, now_dt.month, now_dt.day, WORK_START_1)
            yield env.timeout((next_8 - now_dt).total_seconds()/3600)
            continue

        if WORK_END_1 <= hr < WORK_START_2:
            next_13 = datetime(now_dt.year, now_dt.month, now_dt.day, WORK_START_2)
            yield env.timeout((next_13 - now_dt).total_seconds()/3600)
            continue

        if hr >= WORK_END_2:
            next_day_8 = datetime(now_dt.year, now_dt.month, now_dt.day, WORK_START_1) + timedelta(days=1)
            yield env.timeout((next_day_8 - now_dt).total_seconds()/3600)
            continue

        # อยู่ในเวลางานแล้ว
        break

def get_time_to_block_end(env):
    now_dt = get_current_datetime(env)
    hr = now_dt.hour
    if WORK_START_1 <= hr < WORK_END_1:
        block_end = datetime(now_dt.year, now_dt.month, now_dt.day, WORK_END_1)
    else:
        block_end = datetime(now_dt.year, now_dt.month, now_dt.day, WORK_END_2)
    return (block_end - now_dt).total_seconds()/3600

# ==============================================
# 7) Machine Resource + Log
# ==============================================
class MachineResource:
    def __init__(self, env, machine_id):
        self.env = env
        self.machine_id = machine_id
        self.resource = simpy.PriorityResource(env, capacity=1)
        self.last_size_run = None  # เก็บ track ว่าล่าสุดผลิตขนาดไหน

production_log = []
queue_counter = 0

def log_event(machine_id, order_id, page_id,
              start_dt, finish_dt,
              bottle_size, produced_qty, duration_hrs):
    """
    เก็บ Log ในรูปแบบเบื้องต้นก่อน
    """
    global queue_counter
    queue_counter += 1
    production_log.append({
        "QueueID": queue_counter,
        "MachineID": machine_id,
        "OrderID": order_id,  # -1 สำหรับงานเผื่อ, -2 สำหรับ Changeover
        "PageID": page_id,
        "startTime": start_dt.strftime("%Y-%m-%d %H:%M:%S"),
        "finishTime": finish_dt.strftime("%Y-%m-%d %H:%M:%S"),
        "status": "Completed",
        "bottleSize": bottle_size,
        "producedQuantity": int(produced_qty),
        "durationHours": round(duration_hrs, 2)
    })

def create_machines(env):
    d = {}
    for mc in machines_info:
        d[mc["machine_id"]] = MachineResource(env, mc["machine_id"])
    return d

# ==============================================
# 8) ฟังก์ชัน check/consume วัตถุดิบ
# ==============================================
def check_raw_materials(item_type, brand, size, quantity):
    if item_type == "bottle":
        # จะใช้ preform 1 ชิ้น/ขวด
        if raw_material_stock["Preform"] < quantity:
            return False
    elif item_type == "water":
        # สุดท้ายจะใช้ preform, label, cap, pe film
        if raw_material_stock["Preform"] < quantity:
            return False
        label_key = f"Label_{brand}"
        if label_key not in raw_material_stock or raw_material_stock[label_key] < quantity:
            return False
        cap_key = f"Cap_{brand}"
        if cap_key not in raw_material_stock or raw_material_stock[cap_key] < quantity:
            return False
        rolls_needed = math.ceil(quantity / 9600)
        if raw_material_stock["PE_film_roll"] < rolls_needed:
            return False
    return True

def consume_raw_materials(item_type, brand, size, quantity):
    if item_type == "bottle":
        raw_material_stock["Preform"] -= quantity
    elif item_type == "water":
        raw_material_stock["Preform"] -= quantity
        raw_material_stock[f"Label_{brand}"] -= quantity
        raw_material_stock[f"Cap_{brand}"]   -= quantity
        rolls_needed = math.ceil(quantity / 9600)
        raw_material_stock["PE_film_roll"]   -= rolls_needed

# ==============================================
# 9) do_split_task: ทำงานจริง แบ่งเป็นบล็อก
# ==============================================
def do_split_task(env, machine_id, order_id, page_id,
                  bottle_size, rate, hours_needed):
    remain = hours_needed
    while remain > 0:
        # ข้ามเวลาจนกว่าจะเข้าสู่ช่วงเวลาทำงาน
        yield from skip_to_next_work_time(env)

        block_left = get_time_to_block_end(env)
        if block_left < MIN_BLOCK_LEFT:
            # ถ้าช่วงเวลาทำงานเหลือน้อยกว่า 0.5 ชม ให้ข้ามไปก่อน
            yield env.timeout(block_left)
            continue

        work_hrs = min(remain, block_left)
        start_dt = get_current_datetime(env)
        yield env.timeout(work_hrs)
        finish_dt = get_current_datetime(env)

        remain -= work_hrs
        produced = rate * work_hrs if rate>0 else 0
        dur_hrs = (finish_dt - start_dt).total_seconds()/3600

        # บันทึก event
        log_event(machine_id, order_id, page_id, start_dt, finish_dt,
                  bottle_size, produced, dur_hrs)

# ==============================================
# 10) Pipeline สำหรับ Orders ลูกค้า
# ==============================================
def customer_pipeline(env, order, machines):
    order_id = order["customer"]
    item_type = order["type"]
    brand = order["brand"]
    size = order["size"]
    qty = order["quantity_bottles"]
    priority = order["priority"]

    # ตรวจสอบวัตถุดิบก่อน
    if not check_raw_materials(item_type, brand, size, qty):
        print(f"** วัตถุดิบไม่พอสำหรับ Order {order_id} => skip **")
        return
    consume_raw_materials(item_type, brand, size, qty)

    # เลือก Pipeline
    if item_type == "water":
        if size == "350 ml":
            pipeline_seq = ["MC002", "MC003", "MC004"]
        else:
            pipeline_seq = ["MC001", "MC003", "MC004"]
    else:
        if size == "350 ml":
            pipeline_seq = ["MC002"]
        else:
            pipeline_seq = ["MC001"]

    remain_qty = qty
    for mc_id in pipeline_seq:
        mc_info = next(m for m in machines_info if m["machine_id"]==mc_id)
        idx = mc_info["bottle_size"].index(size)
        rate = mc_info["speed"][idx]
        cng = mc_info["changeover_time"][idx]

        mc_res = machines[mc_id]
        with mc_res.resource.request(priority=priority) as req:
            yield req
            # === ทำ Changeover แยกเป็น orderID = -2 ===
            if mc_res.last_size_run != size:
                if mc_res.last_size_run is not None:
                    yield env.process(
                        do_split_task(env, mc_id, -2, "Changeover", size, 0, cng)
                    )
                mc_res.last_size_run = size

            hours_needed = remain_qty / rate
            yield env.process(
                do_split_task(env, mc_id, order_id, mc_id, size, rate, hours_needed)
            )

# ==============================================
# 11) Pipeline สำหรับการผลิต “เผื่อ”
# ==============================================
def buffer_pipeline(env, item, machines, priority):
    order_id = -1  # หมายถึงงานผลิตเพื่อ buffer
    item_type = item["type"]
    brand = item["brand"]
    size = item["size"]

    can_store = item["want_to_store"] - item["current_store"]
    if can_store <= 0:
        return

    # ตรวจสอบวัตถุดิบ
    if not check_raw_materials(item_type, brand, size, can_store):
        return

    # ตัดสต็อก
    consume_raw_materials(item_type, brand, size, can_store)

    # Pipeline
    if item_type == "water":
        if size == "350 ml":
            pipeline_seq = ["MC002", "MC003", "MC004"]
        else:
            pipeline_seq = ["MC001", "MC003", "MC004"]
    else:
        if size == "350 ml":
            pipeline_seq = ["MC002"]
        else:
            pipeline_seq = ["MC001"]

    remain_qty = can_store
    for mc_id in pipeline_seq:
        mc_info = next(m for m in machines_info if m["machine_id"]==mc_id)
        idx = mc_info["bottle_size"].index(size)
        rate = mc_info["speed"][idx]
        cng = mc_info["changeover_time"][idx]

        mc_res = machines[mc_id]
        with mc_res.resource.request(priority=priority) as req:
            yield req
            if mc_res.last_size_run != size:
                if mc_res.last_size_run is not None:
                    yield env.process(
                        do_split_task(env, mc_id, -2, "Changeover", size, 0, cng)
                    )
                mc_res.last_size_run = size

            hours_needed = remain_qty / rate
            yield env.process(
                do_split_task(env, mc_id, order_id, f"Buffer-{mc_id}", size, rate, hours_needed)
            )

    item["current_store"] += can_store

# ==============================================
# 12) Process หลักในการผลิตเผื่อ
# ==============================================
def buffer_production_flow(env, machines):
    done = False
    while not done:
        done = True
        for bf in sorted(buffer_priority, key=lambda x: x["priority"]):
            bf_type = bf["type"]
            bf_size = bf["size"]
            bf_brand = bf["brand"]
            item = next((it for it in buffer_config
                         if it["type"]==bf_type and it["size"]==bf_size and it["brand"]==bf_brand),
                        None)
            if item is None:
                continue

            need = item["want_to_store"] - item["current_store"]
            if need>0:
                old_val = item["current_store"]
                yield env.process(buffer_pipeline(env, item, machines, priority=9999))
                if item["current_store"]>old_val:
                    done = False

# ==============================================
# 13) run_simulation
# ==============================================
def run_simulation():
    env = simpy.Environment()
    machines = create_machines(env)

    # สั่งผลิตตามลำดับ priority ของ order
    sorted_orders = sorted(customer_orders, key=lambda x: x["priority"])
    for od in sorted_orders:
        env.process(customer_pipeline(env, od, machines))

    # สั่งผลิต buffer
    env.process(buffer_production_flow(env, machines))

    # run จนถึง 2000 ชั่วโมง (เผื่อพอสำหรับคำสั่งทั้งหมด)
    env.run(until=2000)

    return production_log

# ==============================================
# ฟังก์ชันเสริม: แปลงโครงสร้าง production_log
# ==============================================
from datetime import datetime

def transform_production_log(prod_log):
    """
    แปลงข้อมูลให้อยู่ใน structure ตามต้องการ
    - orderID == -1 → "ผลิตเผื่อ"
    - orderID == -2 → "เปลี่ยนขนาด"
    - Mapping machineID (MC001 → MC1, etc.)
    - แก้ format ของเวลาเป็น d/m/YYYY HH:MM
    - ตัด space ออกจาก bottleSize เช่น "600 ml" -> "600ml"
    """

    # Mapping ค่าของ MachineID
    machine_map = {
        "MC001": "MC1",
        "MC002": "MC2",
        "MC003": "MC3",
        "MC004": "MC4"
    }

    new_log = []
    for row in prod_log:
        # แปลง string "YYYY-mm-dd HH:MM:SS" เป็น datetime object
        start_dt = datetime.strptime(row["startTime"], "%Y-%m-%d %H:%M:%S")
        finish_dt = datetime.strptime(row["finishTime"], "%Y-%m-%d %H:%M:%S")

        # format วัน-เวลาให้เป็น d/m/YYYY HH:MM
        start_time_str  = f"{start_dt.day}/{start_dt.month}/{start_dt.year} {start_dt.hour:02d}:{start_dt.minute:02d}"
        finish_time_str = f"{finish_dt.day}/{finish_dt.month}/{finish_dt.year} {finish_dt.hour:02d}:{finish_dt.minute:02d}"

        # แปลง orderID ตามเงื่อนไขที่กำหนด
        if row["OrderID"] == -1:
            order_id = "ผลิตเผื่อ"
        elif row["OrderID"] == -2:
            order_id = "เปลี่ยนขนาด"
        else:
            order_id = row["OrderID"]

        new_log.append({
            "queueID": row["QueueID"],
            "machineID": machine_map.get(row["MachineID"], row["MachineID"]),  # Map ค่า MachineID
            "orderID": order_id,  # ใช้ค่าเปลี่ยนชื่อ orderID
            "pageNumber": 1,  # Fix เป็น 1 ตามที่กำหนด
            "status": row["status"],  
            "startTime": start_time_str,
            "finishTime": finish_time_str,
            "bottleSize": row["bottleSize"].replace(" ", ""),  # ตัด space ใน bottleSize เช่น "600 ml" -> "600ml"
            "producedQuantity": row["producedQuantity"],
        })

    return new_log




# ==============================================
# 14) FastAPI
# ==============================================
app = FastAPI()
app.add_middleware(
    CORSMiddleware,
    allow_origins=["http://localhost:5173"],  # หรือใช้ ["*"] ถ้าต้องการอนุญาตทุกที่
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)

@app.post("/run-simulation/")
def run_simulation_endpoint():
    # เรียก simulation
    prod_log = run_simulation()
    # แปลง log ให้ได้โครงสร้างตามที่ต้องการ
    transformed_log = transform_production_log(prod_log)
    return {
        "message": "Simulation completed",
        "production_log": transformed_log,
        "buffer_config": buffer_config,
        "raw_material_stock": raw_material_stock
    }
