import { useLoadingStore } from './loading'
import { ref } from 'vue'
import { defineStore } from 'pinia'
import userService from '@/services/user'
import type { User } from '@/types/User'
import { useMessageStore } from './message'

export const useUserStore = defineStore('user', () => {
  const loadingStore = useLoadingStore()
  const messageStore = useMessageStore()
  const users = ref<User[]>([])

  const initialUser: User & { files?: File[], previewImage?: string } = {
    email: '',
    password: '',
    name: '',
    gender: 'male',
    role: { id: 1, name: 'user' },
    image: 'noimage.jpg',
    files: [],
    previewImage: '' // ใช้สำหรับแสดงภาพก่อนอัปโหลด
  }
  
  const editedUser = ref<User & { files?: File[], previewImage?: string }>(JSON.parse(JSON.stringify(initialUser)))
  const currentUser = ref<User | null>(null)

  async function getUser(id: number) {
    loadingStore.doLoad()
    const res = await userService.getUser(id)
    editedUser.value = res.data
    editedUser.value.previewImage = `http://localhost:4000/images/users/${res.data.image}` // โหลดรูปเก่า
    loadingStore.finish()
  }

  async function getUsers() {
    loadingStore.doLoad()
    const res = await userService.getUsers()
    users.value = res.data
    loadingStore.finish()
  }

  async function saveUser() {
    try {
      loadingStore.doLoad()
      const user = editedUser.value

      if (!user.id) {
        console.log('Adding user:', JSON.stringify(user))
        await userService.addUser(user)
      } else {
        console.log('Updating user:', JSON.stringify(user))
        await userService.updateUser(user)
      }

      await getUsers()
      loadingStore.finish()
    } catch (e: any) {
      messageStore.showMessage(e.message)
      loadingStore.finish()
    }
  }

  async function deleteUser() {
    if (!editedUser.value.id) return

    loadingStore.doLoad()
    await userService.delUser(editedUser.value.id)
    await getUsers()
    loadingStore.finish()
  }

  function clearForm() {
    editedUser.value = JSON.parse(JSON.stringify(initialUser))
  }

  // 🟢 อัปเดตรูปภาพตอนเลือกไฟล์
  function onFileChange(event: Event) {
    const input = event.target as HTMLInputElement
    if (input.files && input.files.length > 0) {
      const file = input.files[0]
      editedUser.value.files = [file]
      editedUser.value.previewImage = URL.createObjectURL(file) // แสดงรูปก่อนอัปโหลด
    }
  }

  return { users, getUsers, saveUser, deleteUser, editedUser, getUser, clearForm, currentUser, onFileChange }
})
