const express = require('express');
const mysql = require('mysql2');
const dotenv = require('dotenv');
const path = require('path');
const session = require('express-session');
const methodOverride = require('method-override');
const authRoutes = require('./routes/authRoutes');
const productRoutes = require('./routes/productRoutes');
const categoryRoutes = require('./routes/categoryRoutes');
const { isAuthenticated } = require('./middleware/auth');

// โหลดค่าตัวแปรจาก .env
dotenv.config();

const app = express();

// MySQL connection
const db = mysql.createConnection({
  host: process.env.DB_HOST,
  user: process.env.DB_USER,
  password: process.env.DB_PASSWORD,
  database: process.env.DB_NAME
});

db.connect((err) => {
  if (err) {
    console.error('Database connection failed:', err.stack);
    return;
  }
  console.log('Connected to MySQL!');
});

// ใช้ express-session
app.use(session({
  secret: 'your-secret-key',
  resave: false,
  saveUninitialized: true
}));

// Middleware to serve static files (CSS, JS, images)
app.use(express.static(path.join(__dirname, 'public')));
app.use(express.urlencoded({ extended: true }));
app.use(methodOverride('_method'));

// View Engine
app.set('view engine', 'ejs');
app.set('views', path.join(__dirname, 'views'));

// ฟังก์ชันตรวจสอบการล็อกอิน (middleware)
function checkLogin(req, res, next) {
  if (!req.session.user) {
    return res.redirect('/auth/login'); // ถ้ายังไม่ล็อกอินให้ไปหน้า login
  }
  next(); // ถ้าผ่านการตรวจสอบให้ไปหน้าอื่นๆ
}

// Routes
app.use('/auth', authRoutes);  // ใช้เส้นทางจาก authRoutes.js
app.use('/products', isAuthenticated, productRoutes);  // ใช้เส้นทางจาก productRoutes.js และต้องล็อกอินก่อน
app.use('/categories', isAuthenticated, categoryRoutes);

// เส้นทางเริ่มต้น จะไปที่หน้า login
app.get('/', (req, res) => {
  res.redirect('/auth/login'); // เมื่อเปิดเว็บไซต์ จะไปที่หน้า login ทันที
});

// Error Handling Route (สำหรับหน้า 404 หากไม่พบเส้นทางที่กำหนด)
app.use((req, res) => {
  res.status(404).send('Page Not Found');
});

// ฟังก์ชัน logout
app.get('/auth/logout', (req, res) => {
    req.session.destroy((err) => {
      if (err) {
        console.error('Error during logout:', err);
        return res.status(500).send('Failed to logout');
      }
      res.redirect('/auth/login'); // หลังจากออกจากระบบให้กลับไปที่หน้า login
    });
  });

// Error handling middleware
app.use((err, req, res, next) => {
    console.error(err.stack);
    res.status(500).send('เกิดข้อผิดพลาดบางอย่าง!');
});

const PORT = process.env.PORT || 3000;
app.listen(PORT, () => {
  console.log(`Server running on port ${PORT}`);
});
