const db = require('../models/db'); // ประกาศเพียงครั้งเดียว

// ฟังก์ชันสำหรับการแสดงสินค้าทั้งหมด
exports.getAllProducts = (req, res) => {
    const query = `
        SELECT p.*, c.name as category_name 
        FROM products p 
        LEFT JOIN categories c ON p.category_id = c.id 
        ORDER BY p.name
    `;
    
    db.query(query, (err, results) => {
        if (err) {
            console.error('Error fetching products:', err);
            return res.render('index', { products: [], error: 'เกิดข้อผิดพลาดในการดึงข้อมูลสินค้า' });
        }
        res.render('index', { products: results });
    });
};

// ฟังก์ชันสำหรับการสร้างสินค้าใหม่
exports.createProduct = (req, res) => {
    const { name, category_id, quantity, price } = req.body;
    const query = 'INSERT INTO products (name, category_id, quantity, price) VALUES (?, ?, ?, ?)';
    
    db.query(query, [name, category_id, quantity, price], (err, result) => {
        if (err) {
            console.error('Error creating product:', err);
            return res.redirect('/products/add?error=เกิดข้อผิดพลาดในการเพิ่มสินค้า');
        }
        res.redirect('/products');
    });
};

// ฟังก์ชันสำหรับแสดงหน้าแก้ไขสินค้า
exports.editProductPage = (req, res) => {
    const productId = req.params.id;
    const productQuery = `
        SELECT p.*, c.name as category_name 
        FROM products p 
        LEFT JOIN categories c ON p.category_id = c.id 
        WHERE p.id = ?
    `;
    const categoryQuery = 'SELECT * FROM categories ORDER BY name';

    db.query(productQuery, [productId], (err, products) => {
        if (err) {
            console.error('Error fetching product:', err);
            return res.redirect('/products?error=เกิดข้อผิดพลาดในการดึงข้อมูลสินค้า');
        }

        if (products.length === 0) {
            return res.redirect('/products?error=ไม่พบสินค้าที่ต้องการแก้ไข');
        }

        db.query(categoryQuery, (err, categories) => {
            if (err) {
                console.error('Error fetching categories:', err);
                return res.render('editProduct', { 
                    product: products[0], 
                    categories: [],
                    error: 'เกิดข้อผิดพลาดในการดึงข้อมูลหมวดหมู่' 
                });
            }
            res.render('editProduct', { product: products[0], categories });
        });
    });
};

// ฟังก์ชันสำหรับการอัพเดทสินค้า
exports.updateProduct = (req, res) => {
    const { id } = req.params;
    const { name, category_id, quantity, price } = req.body;
    const query = 'UPDATE products SET name = ?, category_id = ?, quantity = ?, price = ? WHERE id = ?';
    
    db.query(query, [name, category_id, quantity, price, id], (err, result) => {
        if (err) {
            console.error('Error updating product:', err);
            return res.redirect(`/products/edit/${id}?error=เกิดข้อผิดพลาดในการแก้ไขสินค้า`);
        }
        res.redirect('/products');
    });
};

// ฟังก์ชันสำหรับการลบสินค้า
exports.deleteProduct = (req, res) => {
    const { id } = req.params;
    
    // ลบข้อมูลในตาราง orders ก่อน
    const deleteOrdersQuery = 'DELETE FROM orders WHERE product_id = ?';
    db.query(deleteOrdersQuery, [id], (err, orderResult) => {
        if (err) {
            console.error('Error deleting orders:', err);
            return res.redirect('/products?error=เกิดข้อผิดพลาดในการลบประวัติการเบิก');
        }

        // หลังจากลบข้อมูลใน orders แล้ว จึงลบสินค้า
        const deleteProductQuery = 'DELETE FROM products WHERE id = ?';
        db.query(deleteProductQuery, [id], (err, productResult) => {
            if (err) {
                console.error('Error deleting product:', err);
                return res.redirect('/products?error=เกิดข้อผิดพลาดในการลบสินค้า');
            }
            res.redirect('/products?success=ลบสินค้าและประวัติการเบิกสำเร็จ');
        });
    });
};

// ฟังก์ชันสำหรับการค้นหาสินค้า
exports.searchProducts = (req, res) => {
    const { searchQuery } = req.query;
    const query = `
        SELECT p.*, c.name as category_name 
        FROM products p 
        LEFT JOIN categories c ON p.category_id = c.id 
        WHERE p.name LIKE ? OR c.name LIKE ?
        ORDER BY p.name
    `;
    const searchTerm = `%${searchQuery}%`;
    
    db.query(query, [searchTerm, searchTerm], (err, results) => {
        if (err) {
            console.error('Error searching products:', err);
            return res.render('index', { products: [], error: 'เกิดข้อผิดพลาดในการค้นหาสินค้า' });
        }
        res.render('index', { products: results, searchQuery });
    });
};

// แสดงหน้าเพิ่มสินค้า
exports.addProductPage = (req, res) => {
    const query = 'SELECT * FROM categories ORDER BY name';
    db.query(query, (err, categories) => {
        if (err) {
            console.error('Error fetching categories:', err);
            return res.render('addProduct', { 
                categories: [], 
                error: 'เกิดข้อผิดพลาดในการดึงข้อมูลหมวดหมู่' 
            });
        }
        res.render('addProduct', { categories });
    });
};

// แสดงหน้ายืนยันการเบิกสินค้า
exports.showWithdrawConfirmation = (req, res) => {
    const { product_id, quantity } = req.body;

    const query = `
        SELECT p.*, c.name as category_name 
        FROM products p 
        LEFT JOIN categories c ON p.category_id = c.id 
        WHERE p.id = ?
    `;
    
    db.query(query, [product_id], (err, results) => {
        if (err) {
            console.error(err);
            return res.redirect('/products?error=เกิดข้อผิดพลาดในการดึงข้อมูลสินค้า');
        }

        if (results.length === 0) {
            return res.redirect('/products?error=ไม่พบสินค้า');
        }

        const product = results[0];
        if (product.quantity < quantity) {
            return res.redirect('/products?error=สินค้าในคลังไม่เพียงพอ');
        }

        res.render('withdrawConfirmation', { 
            product, 
            quantity: parseInt(quantity)
        });
    });
};

// ยืนยันการเบิกสินค้า
exports.confirmWithdraw = (req, res) => {
    const { product_id, quantity } = req.body;

    const query = 'SELECT * FROM products WHERE id = ?';
    db.query(query, [product_id], (err, result) => {
        if (err) {
            console.error(err);
            return res.redirect('/products?error=เกิดข้อผิดพลาดในการเบิกสินค้า');
        }

        if (result.length > 0) {
            const product = result[0];

            if (product.quantity < quantity) {
                return res.redirect('/products?error=สินค้าในคลังไม่เพียงพอ');
            }

            const updateQuery = 'UPDATE products SET quantity = quantity - ? WHERE id = ?';
            db.query(updateQuery, [quantity, product_id], (err, updateResult) => {
                if (err) {
                    console.error(err);
                    return res.redirect('/products?error=เกิดข้อผิดพลาดในการอัพเดทจำนวนสินค้า');
                }

                res.redirect('/products?success=เบิกสินค้าสำเร็จ');
            });
        } else {
            res.redirect('/products?error=ไม่พบสินค้า');
        }
    });
};
