require('dotenv').config();
const express = require('express');
const session = require('express-session');
const bodyParser = require('body-parser');
const bcrypt = require('bcryptjs');
const mysql = require('mysql2');

const app = express();

// อ่านค่า PORT จาก .env ถ้าไม่มีให้ใช้ 3000
const port = process.env.PORT;

// ตั้งค่า view engine เป็น EJS
app.set('view engine', 'ejs');

// ใช้ body-parser สำหรับอ่านข้อมูลจาก form
app.use(bodyParser.urlencoded({ extended: false }));

// ใช้ไฟล์ static (ถ้ามี CSS, รูปภาพ) จากโฟลเดอร์ public
app.use(express.static('public'));

// ตั้งค่า session โดยอ่าน SESSION_SECRET จาก .env
app.use(session({
  secret: process.env.SESSION_SECRET || 'default_secret_key',
  resave: false,
  saveUninitialized: false
}));

// สร้างการเชื่อมต่อกับฐานข้อมูล MySQL โดยอ่านค่าจาก .env
const connection = mysql.createConnection({
  host: process.env.DB_HOST || 'localhost',
  user: process.env.DB_USER || 'root',
  password: process.env.DB_PASS || '',
  database: process.env.DB_NAME || 'cloud_project_db',
  port: process.env.DB_PORT || 3306
});

connection.connect(err => {
  if (err) {
    console.error('Error connecting to MySQL:', err);
    process.exit(1);
  }
  console.log('Connected to MySQL');
});

// Middleware ตรวจสอบการเข้าสู่ระบบ (ใช้สำหรับหน้าที่ต้องการให้ login)
function isAuthenticated(req, res, next) {
  if (req.session.userId) {
    return next();
  }
  res.redirect('/login');
}

// --------------------
// หน้าแรก
// --------------------
app.get('/', (req, res) => {
  // ไม่ว่าจะล็อกอินหรือไม่ ให้ redirect ไปที่ '/dashboard' เสมอ
  res.redirect('/dashboard');
});

// --------------------
// Register Routes
// --------------------
app.get('/register', (req, res) => {
  res.render('register', { error: null });
});

app.post('/register', async (req, res) => {
  const { username, email, password } = req.body;
  try {
    const hashedPassword = await bcrypt.hash(password, 10);
    connection.query(
      'INSERT INTO users (username, email, password) VALUES (?, ?, ?)',
      [username, email, hashedPassword],
      (err) => {
        if (err) {
          console.error(err);
          // ตรวจสอบว่าเป็น ER_DUP_ENTRY (ข้อมูลซ้ำ) หรือไม่
          if (err.code === 'ER_DUP_ENTRY') {
            return res.render('register', { 
              error: 'Username already exists. Please choose another one.' 
            });
          }
          return res.render('register', { error: 'Error registering user' });
        }
        res.redirect('/login');
      }
    );
  } catch (error) {
    console.error(error);
    res.render('register', { error: 'Error processing registration' });
  }
});

// --------------------
// Login Routes
// --------------------
app.get('/login', (req, res) => {
  res.render('login', { error: null });
});

app.post('/login', (req, res) => {
  const { userInput, password } = req.body;
  connection.query(
    'SELECT * FROM users WHERE username = ? OR email = ?',
    [userInput, userInput],
    async (err, results) => {
      if (err) {
        console.error(err);
        return res.render('login', { error: 'Something went wrong, please try again.' });
      }
      if (results.length === 0) {
        return res.render('login', { error: 'User not found.' });
      }
      const user = results[0];
      const match = await bcrypt.compare(password, user.password);
      if (match) {
        req.session.userId = user.id;
        req.session.username = user.username;
        res.redirect('/dashboard');
      } else {
        res.render('login', { error: 'Incorrect password.' });
      }
    }
  );
});

// --------------------
// Dashboard (อนุญาตให้ Guest เข้าดูได้)
// --------------------
app.get('/dashboard', (req, res) => {
  // รับค่าค้นหาจาก query string ?q=
  const searchQuery = req.query.q;
  let sql = `
    SELECT projects.*, users.username AS ownerName
    FROM projects
    JOIN users ON projects.user_id = users.id
    ORDER BY projects.id DESC
  `;

  // ถ้ามีค่า searchQuery ให้เพิ่มเงื่อนไข WHERE เพื่อค้นหาตาม project_name หรือ description
  const params = [];
  if (searchQuery) {
    sql = `
      SELECT projects.*, users.username AS ownerName
      FROM projects
      JOIN users ON projects.user_id = users.id
      WHERE projects.project_name LIKE ?
         OR projects.description LIKE ?
      ORDER BY projects.id DESC
    `;
    params.push(`%${searchQuery}%`, `%${searchQuery}%`);
  }

  connection.query(sql, params, (err, projects) => {
    if (err) {
      console.error(err);
      return res.send('Error fetching projects');
    }
    // ถ้ายังไม่ได้ login ให้แสดงชื่อ Guest
    const username = req.session.username || 'Guest';
    const userId = req.session.userId || null;

    res.render('dashboard', {
      username,
      userId,
      projects,
      query: searchQuery || '',
      currentPage: 'dashboard'
    });
  });
});

// -----------------------------------------------------
// 1) CREATE Project (ต้อง login)
app.get('/project/create', isAuthenticated, (req, res) => {
  res.render('project-create', { username: req.session.username });
});

app.post('/project/create', isAuthenticated, (req, res) => {
  const { project_name, description } = req.body;
  connection.query(
    'INSERT INTO projects (user_id, project_name, description) VALUES (?, ?, ?)',
    [req.session.userId, project_name, description],
    (err) => {
      if (err) {
        console.error(err);
        return res.send('Error creating project');
      }
      res.redirect('/dashboard');
    }
  );
});

// -----------------------------------------------------
// 2) EDIT Project (ต้อง login)
app.get('/project/edit/:id', isAuthenticated, (req, res) => {
  const projectId = req.params.id;
  connection.query(
    'SELECT * FROM projects WHERE id = ? AND user_id = ?',
    [projectId, req.session.userId],
    (err, results) => {
      if (err) {
        console.error(err);
        return res.send('Error fetching project');
      }
      if (results.length === 0) {
        return res.send('Project not found or you are not the owner');
      }
      res.render('project-edit', {
        username: req.session.username,
        project: results[0]
      });
    }
  );
});

app.post('/project/edit/:id', isAuthenticated, (req, res) => {
  const projectId = req.params.id;
  const { project_name, description } = req.body;
  connection.query(
    'UPDATE projects SET project_name = ?, description = ? WHERE id = ? AND user_id = ?',
    [project_name, description, projectId, req.session.userId],
    (err) => {
      if (err) {
        console.error(err);
        return res.send('Error updating project');
      }
      res.redirect('/dashboard');
    }
  );
});

// -----------------------------------------------------
// 3) DELETE Project (ต้อง login)
app.get('/project/delete/:id', isAuthenticated, (req, res) => {
  const projectId = req.params.id;
  connection.query(
    'DELETE FROM projects WHERE id = ? AND user_id = ?',
    [projectId, req.session.userId],
    (err) => {
      if (err) {
        console.error(err);
        return res.send('Error deleting project');
      }
      res.redirect('/dashboard');
    }
  );
});

// -----------------------------------------------------
// 4) VIEW Project + Comments (ต้อง login)
app.get('/project/:id', isAuthenticated, (req, res) => {
  const projectId = req.params.id;
  connection.query(
    `SELECT projects.*, users.username AS ownerName
     FROM projects
     JOIN users ON projects.user_id = users.id
     WHERE projects.id = ?`,
    [projectId],
    (err, results) => {
      if (err) {
        console.error(err);
        return res.send('Error fetching project');
      }
      if (results.length === 0) {
        return res.send('Project not found');
      }
      const project = results[0];

      connection.query(
        `SELECT comments.*, users.username AS commentUser
         FROM comments
         JOIN users ON comments.user_id = users.id
         WHERE comments.project_id = ?
         ORDER BY comments.id DESC`,
        [projectId],
        (err2, comments) => {
          if (err2) {
            console.error(err2);
            return res.send('Error fetching comments');
          }
          res.render('project-view', {
            username: req.session.username,
            userId: req.session.userId,
            project,
            comments
          });
        }
      );
    }
  );
});

// -----------------------------------------------------
// 5) CREATE Comment (ต้อง login)
app.post('/project/:id/comment', isAuthenticated, (req, res) => {
  const projectId = req.params.id;
  const { comment } = req.body;
  connection.query(
    'INSERT INTO comments (project_id, user_id, comment) VALUES (?, ?, ?)',
    [projectId, req.session.userId, comment],
    (err) => {
      if (err) {
        console.error(err);
        return res.send('Error creating comment');
      }
      res.redirect(`/project/${projectId}`);
    }
  );
});

// -----------------------------------------------------
// 6) DELETE Comment (เฉพาะเจ้าของ comment, ต้อง login)
app.get('/comment/delete/:id', isAuthenticated, (req, res) => {
  const commentId = req.params.id;
  const sql = `
    SELECT * FROM comments
    WHERE id = ?
    AND user_id = ?
  `;
  connection.query(sql, [commentId, req.session.userId], (err, results) => {
    if (err) {
      console.error(err);
      return res.send('Error finding comment');
    }
    if (results.length === 0) {
      return res.send('Comment not found or you are not the owner');
    }
    const commentRow = results[0];
    connection.query('DELETE FROM comments WHERE id = ?', [commentId], (err2) => {
      if (err2) {
        console.error(err2);
        return res.send('Error deleting comment');
      }
      res.redirect(`/project/${commentRow.project_id}`);
    });
  });
});

// --------------------
// Logout
// --------------------
app.get('/logout', (req, res) => {
  req.session.destroy();
  res.redirect('/dashboard');
});

// เริ่มเซิร์ฟเวอร์
app.listen(port, '0.0.0.0', () => {
  console.log(`Server running on http://localhost:${port}`);
});
